<?php
/**
 * Class for logging events and errors.
 *
 * @package Divimode_UI
 */

namespace DiviAreasPro;

/**
 * The logger class.
 *
 * The logger creates the new sub-directory "divimode-logs" in the uploads folder and
 * saves the log output into separate .log files: Each log-level has its own file
 * (e.g. "info.log", "error.log", "debug.log")
 *
 * The filename of the .log file also includes a timestamp to create a new file each
 * month, i.e. "info-2020-08.log" for the August 2020 info log file.
 *
 * A logfile can be deleted without consequences at any time (via FTP, etc).
 *
 * Usage:
 *
 * >     // Initialize the logger.
 * >     DM_Logger::init('divi-areas', WP_DEBUG_LOG);
 * >
 * >     // Add new line to the log file:
 * >     DM_Logger::debug( 'message', $var, 123, [1, 2, 3] );
 * >     DM_Logger::error( 'message', $var, 123, [1, 2, 3] );
 * >
 * >     // Explicitely log to a certain file:
 * >     DM_Logger::log( 'debug', 'message', $var, 123, [1, 2, 3] );
 * >     DM_Logger::log( 'cron', 'message', $var, 123, [1, 2, 3] );
 * >     DM_Logger::log( 'custom', 'message', $var, 123, [1, 2, 3] );
 */
class DM_Logger {

	/**
	 * The sub-directory inside the uploads folder that stores the log files.
	 *
	 * @since 2.0.0
	 * @var string
	 */
	private $subdir = 'divimode-logs';

	/**
	 * Whether logging is enabled or not.
	 *
	 * @since 2.0.0
	 * @var bool
	 */
	private $enabled = false;

	/**
	 * A static hash that is generated by the constructor and prefixed to all log
	 * entries. The purpose of the hash is to identify log entries that belong to the
	 * same HTTP request.
	 *
	 * @since 2.0.0
	 * @var string
	 */
	private $request_hash = '';

	/**
	 * A timestamp that is set by the constructor to measure the request duration.
	 *
	 * @since 2.0.0
	 * @var int
	 */
	private $request_start = 0;

	/**
	 * Contains a list of all log-types and their entries. This data is written to the
	 * log file during the shutdown action.
	 *
	 * @since 2.0.0
	 * @var array
	 */
	private $logs = [];

	/**
	 * Initializes and configures the logger class.
	 *
	 * @since 2.0.0
	 * @param string    $subdir See DM_logger::set_subdir().
	 * @param bool|null $enabled See DM_Logger::set_state().
	 * @return void
	 */
	public function __construct( $subdir = '', $enabled = null ) {
		$this->request_start = microtime( true );

		$this->set_subdir( $subdir );
		$this->set_state( $enabled );

		add_action( 'shutdown', [ $this, 'write_logs' ] );
	}

	/**
	 * Public getter to query the current enabled-state.
	 *
	 * @since 2.0.0
	 * @return bool
	 */
	public function is_enabled() {
		return $this->enabled;
	}

	/**
	 * Changes the enabled-state of the logger and returns the previous state.
	 *
	 * @since 2.0.0
	 * @param bool|null $enabled Optional. Explicitely enable or disable logging.
	 *                           Default is null, which will use either DIVIMODE_DEBUG
	 *                           or WP_DEBUG_LOG flag.
	 * @return bool The previous state of the logger.
	 */
	public function set_state( $enabled = null ) {
		$orig_state = $this->enabled;

		if ( null === $enabled ) {
			if ( defined( 'DIVIMODE_DEBUG' ) ) {
				$enabled = DIVIMODE_DEBUG;
			} else {
				$enabled = defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG;
			}
		}

		$this->enabled = (bool) $enabled;

		return $orig_state;
	}

	/**
	 * Changes the sub-directory for the log output and returns the previous value.
	 *
	 * @since 2.0.0
	 * @param string $subdir Optional. Custom directory inside the uploads dir that
	 *                       will hold the log files. Default is 'divimode-logs'.
	 * @return string The previous output directory.
	 */
	public function set_subdir( $subdir = '' ) {
		$orig_dir = $this->subdir;

		$subdir = sanitize_key( $subdir );
		if ( ! $subdir ) {
			$subdir = 'divimode-logs';
		}

		$this->subdir = $subdir;

		return $orig_dir;
	}

	/**
	 * Writes the log-contents to the respective log files and clears the log-contents
	 * cache. This function can be called anytime to flush the log cache and is always
	 * called in the final shutdown action.
	 *
	 * @since 2.0.0
	 * @return void
	 */
	public function write_logs() {
		$suffix = '';

		if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
			$suffix .= '-ajax';
		} elseif ( defined( 'DOING_CRON' ) && DOING_CRON ) {
			$suffix .= '-cron';
		} elseif ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
			// Note: REST_REQUEST is defined in/after the `parse_request` action.
			$suffix .= '-rest';
		}

		foreach ( $this->logs as $type => $lines ) {
			if ( ! count( $lines ) ) {
				continue;
			}

			if ( 'error' === $type ) {
				$logfile = $this->get_log_path( $type );
			} else {
				$logfile = $this->get_log_path( $type . $suffix );
			}

			if ( $logfile ) {
				$message = implode( "\n", $lines ) . "\n";
				error_log( $message, 3, $logfile ); // phpcs:ignore
			}

			$this->logs[ $type ] = [];
		}
	}

	/**
	 * Initializes the log-cache for the specified type.
	 *
	 * phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	 * phpcs:disable ET.Sniffs.ValidatedSanitizedInput.InputNotSanitized
	 * phpcs:disable WordPress.Security.ValidatedSanitizedInput.MissingUnslash
	 * phpcs:disable WordPress.Security.NonceVerification.Missing
	 *
	 * @since 2.0.0
	 * @param string $type The log-type (debug, error, etc).
	 * @return void
	 */
	protected function prepare_log_type( $type ) {
		$this->logs[ $type ] = [];

		$this->logs[ $type ][] = sprintf(
			"\n[%s %s]\t[REQ %s]\tURI: %s%s",
			gmdate( 'Y-m-d' ),
			gmdate( 'H:i:s' ),
			$this->get_request_hash(),
			isset( $_SERVER['REQUEST_URI'] ) ? $_SERVER['REQUEST_URI'] : '/',
			isset( $_POST['action'] ) ? "\tAction: " . $_POST['action'] : ''
		);
	}

	/**
	 * Returns the current request hash to identify log entries that are generated
	 * during the same HTTP request. The hash is random and generated during the first
	 * call to this function. This hash is used to reference related log entries in
	 * different log files; i.e., find debug entries for an error.
	 *
	 * @since 2.0.0
	 * @return string The request hash ("$" + 9 random uppercase chars).
	 */
	protected function get_request_hash() {
		if ( ! $this->request_hash ) {
			$prefix = wp_rand( 36, 1295 );
			$hash   = crc32( microtime() . uniqid( wp_rand() ) );

			$this->request_hash = '$' . str_pad(
				strtoupper(
					base_convert( $prefix, 10, 36 ) .
					base_convert( $hash, 10, 36 )
				),
				9,
				'0'
			);
		}
		return $this->request_hash;
	}

	/**
	 * Returns the absolute path to the current logfile.
	 *
	 * @since 2.0.0
	 * @param string $type The log file type (error, info).
	 * @return string
	 */
	protected function get_log_path( $type ) {
		$file       = $this->get_filename( $type );
		$upload_dir = wp_upload_dir();
		$folder     = $upload_dir['basedir'] . DIRECTORY_SEPARATOR . $this->subdir;

		if ( ! is_dir( $folder ) ) {
			wp_mkdir_p( $folder );
		}
		if ( ! is_dir( $folder ) ) {
			return false;
		}

		return $folder . DIRECTORY_SEPARATOR . $file;
	}

	/**
	 * Returns the filename of the current logfile without a directory.
	 *
	 * @since 2.0.0
	 * @param string $type The log file type (error, info).
	 * @return string
	 */
	public function get_filename( $type = 'info' ) {
		$file = sprintf(
			'%s_%s.log',
			strtolower( trim( $type ) ),
			gmdate( 'Y-m' )
		);

		return $file;
	}

	/**
	 * Formulates the full log entry and returns a single string.
	 *
	 * phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_print_r
	 *
	 * @since 2.0.0
	 * @param array $args An array of all arguments that were passed to the logger.
	 * @return string
	 */
	protected function get_log_message( $args ) {
		$parts = [];

		foreach ( $args as $message ) {
			if ( is_bool( $message ) ) {
				$message_ = $message ? 'true' : 'false';
			} elseif ( is_scalar( $message ) ) {
				$message_ = (string) $message;
			} elseif ( is_array( $message ) ) {
				$message_ = str_replace( "\n", "\n\t", print_r( $message, true ) );
			} else {
				$message_ = wp_json_encode( $message );
			}

			$parts[] = $message_;
		}

		return sprintf(
			"\t%03dms:\t%s",
			microtime( true ) - $this->request_start,
			trim( implode( ' | ', $parts ) )
		);
	}

	/**
	 * Read the log file contents of the given file.
	 *
	 * Can be used to output log contents in the wp-admin page.
	 *
	 * @since 2.0.0
	 * @param string $type The logfile type (info, error, cron, ...).
	 */
	public function get_log_contents( $type ) {
		$type    = strtolower( trim( $type ) );
		$logfile = $this->get_log_path( $type );

		if ( ! is_file( $logfile ) ) {
			return '';
		}

		if ( filesize( $logfile ) > 800000 ) {
			return 'The log-file is too big to display here.';
		} else {
			// phpcs:ignore
			return file_get_contents( $logfile );
		}
	}

	/**
	 * Logger: Log to the specified file.
	 *
	 * @since 2.0.0
	 * @param string $type       The logfile type (debug, error, cron, ...).
	 * @param mixed  ...$message A single variable to log or multiple parameters.
	 */
	public function log( $type, ...$message ) {
		if ( ! $this->enabled ) {
			return;
		}

		$type = sanitize_key( $type );

		if ( ! isset( $this->logs[ $type ] ) ) {
			$this->prepare_log_type( $type );
		}
		$this->logs[ $type ][] = $this->get_log_message( $message );
	}

	/**
	 * Logger: Create new debug log entry in either of the debug.log, debug-cron.log
	 * or debug-ajax.log files.
	 *
	 * @since 2.0.0
	 * @param mixed ...$message A single variable to log or multiple parameters.
	 */
	public function debug( ...$message ) {
		if ( ! $this->enabled ) {
			return;
		}

		if ( ! isset( $this->logs['debug'] ) ) {
			$this->prepare_log_type( 'debug' );
		}
		$this->logs['debug'][] = $this->get_log_message( $message );
	}

	/**
	 * Logger: Create new error log entry in the error.log file.
	 *
	 * @since 2.0.0
	 * @param mixed ...$message A single variable to log or multiple parameters.
	 */
	public function error( ...$message ) {
		if ( ! $this->enabled ) {
			return;
		}

		if ( ! isset( $this->logs['error'] ) ) {
			$this->prepare_log_type( 'error' );
		}
		$this->logs['error'][] = $this->get_log_message( $message );
	}
}
